#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "aftt.h"
#include "afusb.h"
#include "cart.h"
#include "config.h"
#include "convert.h"
#include "globals.h"
#include "util.h"

const char aftt_header_magic[] = "RRBY64AF";

/* -------------------------------------------------------------------------- */

static int aftt_to_crt_inloop(int argc, char** argv, FILE *fd_out, int fd_usb)
{
    FILE *fd = NULL;
    int rc = 0;
    int i = 0;

    while ((i < argc) && (rc == 0)) {
        util_message("Injecting AFTT file %i/%i '%s'...", i + 1, argc, argv[i]);
        fd = fopen(argv[i], "rb");
        if (fd == NULL) {
            util_error("Failed to open file `%s'!", argv[i]);
            return -1;
        }

        rc = aftt_run(fd, fd_out, 0, fd_usb);
        fclose(fd);

        ++i;
    }

    return rc;
}

int aftt_to_afusb(int argc, char** argv)
{
    FILE *fd_out = NULL;
    int fd_usb = -1;

    if (argc == 2) {
        fd_out = fopen(argv[1], "wb");
        if (fd_out == NULL) {
            return -1;
        }
    }

    fd_usb = afusb_get_fd();

    if (fd_usb == -1) {
        return -1;
    }

    return aftt_to_crt_inloop(1, &argv[0], fd_out, fd_usb);
}

int aftt_to_crt(int argc, char** argv)
{
    int i = 0;

    if (((argc - i) > 0) && (aftt_to_crt_inloop(argc - i, &argv[i], NULL, -1) < 0)) {
        return -1;
    }

    return 0;
}

int aftt_to_bin(int argc, char** argv)
{
    if ((argc > 1) && (aftt_to_crt_inloop(argc - 1, &argv[0], NULL, -1) < 0)) {
        return -1;
    }

    if (flash_prog_end <= flash_prog_low) {
        util_warning("nothing to save, skipping.");
        return 0;
    }

    return bin_save(argv[argc - 1], flash_prog_low, flash_prog_end - 1);
}

/* -------------------------------------------------------------------------- */

static int get_filetype_params(unsigned char type, unsigned int *addr, const char **typetext, size_t *max_size_out)
{
    size_t max_size = 0;

    switch (type) {
        case AFTT_TYPE_D64:
            *addr = 0x40000;
            *typetext = "D64";
            max_size = 0x2ab00; /* FIXME */
            break;
        case AFTT_TYPE_PRG:
            *addr = 0x10000;
            *typetext = "PRG";
            max_size = 0xd000; /* real max slightly smaller */
            break;
        default:
            return -1;
    }

    if (max_size_out) {
        *max_size_out = max_size;
    }

    return 0;
}

static int load_file(const char *filename, const char *filetype_string, unsigned char *data, size_t *datasize_out, size_t max_size, int interleave)
{
    FILE *fd = NULL;
    int i;
    unsigned char dummy;
    size_t datasize, chunksize, dest, gotsize;

    util_message("Loading %s file '%s' with interleave $%06x...", filetype_string, filename, interleave);

    fd = fopen(filename, "rb");

    if (fd == NULL) {
        util_error("problems opening file '%s'!", filename);
        return -1;
    }

    if (interleave != 0) {
        chunksize = 0x2000;
    } else {
        chunksize = max_size;
    }

    i = 0;
    datasize = 0;
    dest = 0;
    while ((datasize < max_size) && !feof(fd)) {
        gotsize = fread(&data[dest], 1, chunksize, fd);

        if (gotsize > 0) {
            datasize += gotsize;
        }

        if (gotsize < chunksize) {
            break;
        }

        i ^= 1;

        if (interleave && (i & 1)) {
            dest += interleave;
        } else {
            dest -= interleave;
            dest += chunksize;
        }
    }

    if (!feof(fd) && (fread(&dummy, 1, 1, fd) > 0)) {
        util_error("file '%s' was longer than %lu bytes!", filename, (unsigned long)max_size);
        fclose(fd);
        return -1;
    }

    fclose(fd);

    if (datasize < 2) {
        util_error("problems reading file '%s'!", filename);
        return -1;
    }

    if (datasize_out != NULL) {
        *datasize_out = datasize;
    }

    return 0;
}

static unsigned char *load_file_set_header(const char *filename, unsigned char type, size_t *datasize_out, unsigned int *addr_out)
{
    size_t max_size, datasize;
    unsigned char *data = NULL;
    const char *filetype_string = "???";
    unsigned int start_addr = 0;
    char *descr = NULL;

    if (get_filetype_params(type, &start_addr, &filetype_string, &max_size) < 0) {
        return NULL;
    }

    data = malloc(max_size + AFTT_HEADER_LEN);

    if (data == NULL) {
        util_error("failed to malloc %lu bytes!", (unsigned long)(max_size + AFTT_HEADER_LEN));
        return NULL;
    }

    if (load_file(filename, filetype_string, &(data[AFTT_HEADER_LEN]), &datasize, max_size, 0)) {
        free(data);
        return NULL;
    }

    /* set up AFTT header */
    memset(data, 0xff, 0x100);
    memcpy(data, aftt_header_magic, 8);
    data[0x08] = type;
    data[0x18] = datasize & 0xff;
    data[0x19] = (datasize >> 8) & 0xff;
    data[0x1a] = (datasize >> 16) & 0xff;
    data[0x1b] = (datasize >> 24) & 0xff;

    /* if the desctiption is empty... */
    if (text_description[0] == '\0') {
        switch (type) {
            case AFTT_TYPE_D64:
                /* get the description from the image itself */
                descr = (char *)&(data[AFTT_HEADER_LEN + 0x16590]);
                break;
            case AFTT_TYPE_PRG:
                descr = "PRG FROM AFTOOL ";
                break;
            default:
                return NULL;
        }
    } else {
        descr = text_description;
    }

    memcpy(&data[0x20], (unsigned char *)descr, 0x10);

    if (datasize_out != NULL) {
        *datasize_out = datasize + AFTT_HEADER_LEN;
    }

    if (addr_out != NULL) {
        *addr_out = start_addr;
    }

    return data;
}

int prg_to_aftt(int argc, char** argv)
{
    int rc = 0;
    size_t datasize = 0;
    unsigned int addr = 0;
    unsigned char *data = NULL;

    if (argc < 2) {
        return -1;
    }

    data = load_file_set_header(argv[0], AFTT_TYPE_PRG, &datasize, &addr);

    if (data == NULL) {
        return -1;
    }

    rc = aftt_save(argv[1], data, datasize, addr);

    free(data);
    return rc;
}

int d64_to_aftt(int argc, char** argv)
{
    int rc = 0;
    size_t datasize = 0;
    unsigned int addr = 0;
    unsigned char *data = NULL;

    if (argc < 2) {
        return -1;
    }

    data = load_file_set_header(argv[0], AFTT_TYPE_D64, &datasize, &addr);

    if (data == NULL) {
        return -1;
    }

    rc = aftt_save(argv[1], data, datasize, addr);

    free(data);
    return rc;
}

int crt_to_aftt(int argc, char** argv)
{
    int offset = -1;
    unsigned int last_addr = 0;
    int romh_too = 0;

    if (argc == 3) {
        if (parse_offset(argv[2], &offset, NULL, NULL) < 0) {
            return -1;
        }
    }

    /* FIXME */
    if (0
        || (anycart_load(argv[0], &offset, &last_addr, &romh_too) < 0)
        || (aftt_save(argv[1], &flash_data[offset], last_addr - offset, offset) < 0)) {
        return -1;
    }

    return 0;
}

/* -------------------------------------------------------------------------- */

int inject_file_to_crt_or_usb(const char *filename_file, unsigned char type, int is_usb)
{
    size_t datasize = 0;
    unsigned int addr = 0;

    if (filename_file != NULL) {
        /* load file from drive */
        unsigned char *data = NULL;

        data = load_file_set_header(filename_file, type, &datasize, &addr);

        if (data == NULL) {
            return -1;
        }

        memcpy(&flash_data[addr], data, datasize);
        free(data);
    } else {
        /* use the file in flash_data */
        char buf[16 + 1];
        int len;
        int load;
        int offset = 0;
        int offset_end;
        const char *typetext = "???";

        if (get_filetype_params(type, &addr, &typetext, NULL) < 0) {
            return -1;
        }

        offset = (int)addr;

        if (!detect_aftt_header(offset, type, &len, buf, &load)) {
            util_error("no %s found at addr $%06x.", typetext, offset);
            return -1;
        }

        datasize = (size_t)(len + AFTT_HEADER_LEN);

        offset += AFTT_HEADER_LEN;
        offset_end = offset + len - 1;

        util_message("Found %s '%s' at $%06x - $%06x.", typetext, buf, offset, offset_end);
    }

    if (is_usb) {
        return aftt_usb_send(flash_data, datasize, addr, 0, 0 /* don't keep old data */, 1);
    }

    return 0;
}

int inject_crt_to_afcrt(int argc, char** argv)
{
    int offset = -1;

    if (argc == 2) {
        if (parse_offset(argv[1], &offset, NULL, NULL) < 0) {
            return -1;
        }
    }

    return anycart_load(argv[0], &offset, NULL, NULL);
}

int inject_bin_to_afcrt(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    int i = 0;

    if (argc == 3) {
        if (afcart_load(argv[0]) < 0) {
            return -1;
        }
        ++i;
    }

    if (parse_offset(argv[i + 1], &offset, &offset_end, &romh_too) < 0) {
        return -1;
    }

    if (load_file(argv[i + 0], "bin", &(flash_data[offset]), NULL, 0x1000000 - offset, romh_too * 0x800000)) {
        return -1;
    }

    return 0;
}

/* -------------------------------------------------------------------------- */

int extract_file_from_crt_or_usb(const char *filename_out, unsigned char type, int is_usb)
{
    char buf[16 + 1];
    int len;
    int load;
    int offset;
    int offset_end;
    unsigned int start_addr = 0;
    const char *typetext = "???";

    if (get_filetype_params(type, &start_addr, &typetext, NULL) < 0) {
        return -1;
    }

    offset = (int)start_addr;
    offset_end = offset + AFTT_HEADER_LEN - 1;

    if (is_usb) {
        /* read the header via USB */
        util_message("Receiving AFTT header...");
        if (aftt_usb_load(offset, offset_end, 0) < 0) {
            return -1;
        }
    }

    if (!detect_aftt_header(offset, type, &len, buf, &load)) {
        util_error("no %s found at addr $%06x.", typetext, offset);
        return -1;
    }

    offset += AFTT_HEADER_LEN;
    offset_end = offset + len - 1;

    util_message("Found %s '%s' at $%06x - $%06x.", typetext, buf, offset, offset_end);

    if (is_usb) {
        /* read the actual file */
        util_message("Receiving %s file...", typetext);
        if (aftt_usb_load(offset, offset_end, 0) < 0) {
            return -1;
        }
    }

    if (filename_out == NULL) {
        /* no output file, just load to image */
        return 0;
    }

    return bin_save(filename_out, offset, offset_end);
}

static int parse_carttype(const char *s, int *offset, int *offset_end, int *romh_too)
{
    if (s == NULL) {
        return -1;
    } else if (strcmp(s, "ar") == 0) {
        *offset = 0x870000;
        *offset_end = 0x877fff;
        *romh_too = 0;
        return CARTRIDGE_ACTION_REPLAY;
    } else if (strcmp(s, "ef") == 0) {
        *offset = 0x080000;
        *offset_end = 0x0fffff;
        *romh_too = 1;
        return CARTRIDGE_EASYFLASH;
    } else if (strcmp(s, "af") == 0) {
        *offset = 0;
        *offset_end = 0x7fffff;
        *romh_too = 1;
        return CARTRIDGE_ALIEN_FLASH;
    } else if (strcmp(s, "at") == 0) {
        *offset = 0;
        *offset_end = 0x00ffff;
        *romh_too = 1;
        return CARTRIDGE_ALIEN_FLASH;
    }

    util_error("unknown CARTTYPE '%s'", s);

    return -1;
}

int get_crtid_and_offsets(int argc, char** argv, int *offset, int *offset_end, int *romh_too)
{
    int crtid = -1;
    int offset_end_l = -1;

    crtid = parse_carttype(argv[0], offset, offset_end, romh_too);

    if (crtid < 0) {
        return -1;
    }

    if (offset_end != NULL) {
        offset_end_l = *offset_end;
    }

    if (argc == 2) {
        if (parse_offset(argv[1], offset, offset_end, romh_too) < 0) {
            return -1;
        }
    }

    if ((offset_end != NULL) && ((*offset_end) < 0)) {
        *offset_end = offset_end_l;
    }

    return crtid;
}

static int save_a_crt_internal(const char *filename, int offset, int offset_end, int romh_too, int crtid)
{
    int rc = -1;

    switch (crtid) {
        case CARTRIDGE_ACTION_REPLAY:
            rc = arcart_save(filename, offset, 4, crtid);
            break;
        case CARTRIDGE_EASYFLASH:
            rc = efcart_save(filename, offset, offset_end);
            break;
        case CARTRIDGE_ALIEN_FLASH:
            rc = afcart_save(filename, offset, offset_end);
            break;
    }

    return rc;
}

int extract_crt_from_afcrt(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    int crtid = -1;

    crtid = get_crtid_and_offsets(argc - 1, &argv[1], &offset, &offset_end, &romh_too);

    if (crtid < 0) {
        return -1;
    }

    return save_a_crt_internal(argv[0], offset, offset_end, romh_too, crtid);
}

int afcrt_to_bin(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;

    if (parse_offset((argc == 2) ? argv[1] : "af", &offset, &offset_end, &romh_too) < 0) {
        return -1;
    }

    if (romh_too) {
        util_error("ROM SLOT -> binary not supported currently.");
        return -1;
    }

    if (offset_end < 0) {
        util_error("end address missing.");
        return -1;
    }

    return bin_save(argv[0], offset, offset_end);
}

int afusb_to_bin(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;

    if (parse_offset(argv[1], &offset, &offset_end, &romh_too) < 0) {
        return -1;
    }

    if (romh_too) {
        util_error("ROM SLOT -> binary not supported currently.");
        return -1;
    }

    if (offset_end < 0) {
        util_error("end address missing.");
        return -1;
    }

    if (aftt_usb_load(offset, offset_end, romh_too) < 0) {
        util_error("problems receiving binary.");
        return -1;
    }

    return bin_save(argv[0], offset, offset_end);
}

/* -------------------------------------------------------------------------- */

int send_crtfile_to_afusb(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    unsigned int last_addr = 0;

    if (argc == 2) {
        if (parse_offset(argv[1], &offset, &offset_end, &romh_too) < 0) {
            return -1;
        }
    }

    if (offset_end >= 0) {
        last_addr = (unsigned int)offset_end;
    }

    if (argv[0] != NULL) {
        if (anycart_load(argv[0], &offset, &last_addr, &romh_too) < 0) {
            return -1;
        }
    } else {
        /* use CRT in flash_data */
    }

    if (offset > last_addr) {
        util_error("Bug: offset %i > last_addr %u (offset_end %i)", offset, last_addr, offset_end);
        return -1;
    }

    return aftt_usb_send(flash_data, last_addr - offset, offset, romh_too, 1, 1);
}

int get_crtfile_from_afusb(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    int crtid = -1;

    crtid = get_crtid_and_offsets(argc - 1, &argv[1], &offset, &offset_end, &romh_too);

    if (aftt_usb_load(offset, offset_end, romh_too) < 0) {
        return -1;
    }

    if (argv[0] != NULL) {
        return save_a_crt_internal(argv[0], offset, offset_end, romh_too, crtid);
    } else {
        /* just keep it in flash_data */
        return 0;
    }
}
