#include <stdio.h>
#include <stdlib.h>

#include "aftt.h"
#include "cart.h"
#include "config.h"
#include "dump.h"
#include "flashcore.h"
#include "globals.h"
#include "util.h"

int dump_aftt(int argc, char** argv)
{
    FILE *fd = NULL;
    int rc = -1;
    int vlevel = 1;

    util_message("Dumping AFTT file %s...", argv[0]);

    if (argc > 1) {
        vlevel = atoi(argv[1]);
        if ((vlevel < 0) || (vlevel > 3)) {
            util_error("unsupported verbosity level %s!", argv[1]);
            return -1;
        }
    }

    if (flashcore_init(vlevel) < 0) {
        return -1;
    }

    fd = fopen(argv[0], "rb");
    if (fd == NULL) {
        util_error("Failed to open file `%s'!", argv[0]);
        return -1;
    }


    rc = aftt_run(fd, NULL, vlevel, -1);

    fclose(fd);

    flashcore_shutdown();

    return rc;
}


static void dump_print_range(int s, int e, int lh, const char *text)
{
    util_message("  %s: $%06x - $%06x (banks %c:$%03x - %c:$%03x)",
            text, s, e,
            lh ? 'H' : 'L',
            (s / 0x2000) & 0x3ff,
            lh ? 'H' : 'L',
            (e / 0x2000) & 0x3ff
            );
}

int detect_cbm80(int offset)
{
    int i;
    unsigned char cbm80[5] = { 0xc3, 0xc2, 0xcd, 0x38, 0x30 };

    for (i = 0; i < 5; ++i) {
        if (flash_data[offset + i + 4] != cbm80[i]) {
            return 0;
        }
    }
    return 1;
}

int detect_bootvec(int offset)
{
    int i;

    for (i = 0; i < 6; ++i) {
        if (flash_data[offset + 0x1ffa + i] != 0xff) {
            return 1;
        }
    }

    return 0;
}

int detect_ar(int offset)
{
    if (offset < 0) {
        offset = 0x870000;
    }

    return detect_cbm80(offset);
}

int dump_afcrt(int argc, char** argv)
{
    int base;
    int i;
    int slot;
    int lh;
    int had_data;
    int has_data;
    int range_start = 0;
    char buf[16 + 1];
    int len;
    int load;
    unsigned char type;

    util_message("Memory ranges:");

    for (slot = 0; slot < 16; ++slot) {
        util_message(" ROM SLOT %i", slot);

        for (lh = 0; lh < 2; ++lh) {
            base = slot * 0x080000 + lh * 0x800000;
            has_data = 0;
            range_start = base;

            for (i = 0; i <= 0x080000; i += 0x2000) {
                had_data = has_data;
                has_data = (i < 0x080000) ? !af_check_empty(&flash_data[base + i]) : !has_data;

                if (has_data && !had_data && (i > 0)) {
                    dump_print_range(range_start, base + i - 1, lh, "empty");
                    range_start = base + i;
                } else if (!has_data && had_data) {
                    dump_print_range(range_start, base + i - 1, lh, " data");
                    range_start = base + i;
                }
            }
        }
    }

    range_start = 0x10000;
    type = AFTT_TYPE_PRG;
    if (detect_aftt_header(range_start, type, &len, buf, &load)) {
        util_message("Found PRG (type $%02x) '%s' at $%06x - $%06x, load to $%04x - $%04x",
                type, buf, range_start, range_start + 0x100 + len - 1, load, load + len - 1 - 2);
    }

    range_start = 0x40000;
    type = AFTT_TYPE_D64;
    if (detect_aftt_header(range_start, type, &len, buf, &load)) {
        util_message("Found D64 (type $%02x) '%s' at $%06x - $%06x",
                type, buf, range_start, range_start + 0x100 + len - 1);

    }

    range_start = 0x870000;
    if (detect_ar(range_start)) {
        util_message("Found CBM80 on the Action Replay image location $%06x", range_start);
    }

    return 0;
}
