/*
 * flashcore.c - Am29F032B Flash emulation.
 *
 * Based on flash040core.c from VICE.
 *
 * Written by
 *  Hannu Nuotio <hannu.nuotio@tut.fi>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "flashcore.h"
#include "globals.h"
#include "util.h"

/* -------------------------------------------------------------------------- */

enum flash040_state_s {
    FLASH040_STATE_READ,
    FLASH040_STATE_MAGIC_1,
    FLASH040_STATE_MAGIC_2,
    FLASH040_STATE_AUTOSELECT,
    FLASH040_STATE_BYTE_PROGRAM,
    FLASH040_STATE_BYTE_PROGRAM_ERROR,
    FLASH040_STATE_ERASE_MAGIC_1,
    FLASH040_STATE_ERASE_MAGIC_2,
    FLASH040_STATE_ERASE_SELECT,
    FLASH040_STATE_CHIP_ERASE,
    FLASH040_STATE_SECTOR_ERASE
};
typedef enum flash040_state_s flash040_state_t;

/* #define FLASH_DEBUG_ENABLED */

#ifdef FLASH_DEBUG_ENABLED
#define FLASH_DEBUG(x) log_debug x
#else
#define FLASH_DEBUG(x)
#endif

struct flash_types_s {
    unsigned char manufacturer_ID;
    unsigned char device_ID;
    unsigned int size;
    unsigned int sector_mask;
    unsigned int sector_size;
    unsigned int sector_shift;
    unsigned int magic_1_addr;
    unsigned int magic_2_addr;
    unsigned int magic_1_mask;
    unsigned int magic_2_mask;
    unsigned char status_toggle_bits;
};
typedef struct flash_types_s flash_types_t;

static flash_types_t flash_type = {
    /* 29F032B */
    0x01, 0x41,
    0x400000,
    0x3f0000, 0x10000, 16,
    0x555,  0x2aa,  0x7ff,  0x7ff,
    0x44
};

typedef struct flash040_context_s {
    unsigned char *flash_data;
    flash040_state_t flash_state;
    flash040_state_t flash_base_state;

    unsigned char program_byte;
    int flash_dirty;

    unsigned int base_addr;
} flash040_context_t;

/* -------------------------------------------------------------------------- */

/* globally visible Flash data */
unsigned char *flash_data = NULL;
unsigned int flash_prog_low;
unsigned int flash_prog_end;

/* context of the 4 Flash chips */
static flash040_context_t af_chip[4] = {
    { NULL, FLASH040_STATE_READ, FLASH040_STATE_READ, 0, 0, 0x000000 },
    { NULL, FLASH040_STATE_READ, FLASH040_STATE_READ, 0, 0, 0x040000 },
    { NULL, FLASH040_STATE_READ, FLASH040_STATE_READ, 0, 0, 0x080000 },
    { NULL, FLASH040_STATE_READ, FLASH040_STATE_READ, 0, 0, 0x0c0000 }
};

static int vlevel = 0;

/* -------------------------------------------------------------------------- */

inline static int flash_magic_1(unsigned int addr)
{
    return ((addr & flash_type.magic_1_mask) == flash_type.magic_1_addr);
}

inline static int flash_magic_2(unsigned int addr)
{
    return ((addr & flash_type.magic_2_mask) == flash_type.magic_2_addr);
}

inline static void flash_erase_sector(flash040_context_t *flash040_context, unsigned int addr)
{
    unsigned int sector_size = flash_type.sector_size;
    unsigned int sector_addr = addr & flash_type.sector_mask;

    if (vlevel > 0) {
        util_message("Flash: Erasing $%06x - $%06x", sector_addr, sector_addr + sector_size - 1);
    }
    memset(&(flash040_context->flash_data[sector_addr]), 0xff, sector_size);
    flash040_context->flash_dirty = 1;
}

inline static void flash_erase_chip(flash040_context_t *flash040_context)
{
    if (vlevel > 0) {
        util_message("Flash: Erasing chip");
    }
    memset(flash040_context->flash_data, 0xff, flash_type.size);
    flash040_context->flash_dirty = 1;
}

inline static int flash_program_byte(flash040_context_t *flash040_context, unsigned int addr, unsigned char byte)
{
    unsigned char old_data = flash040_context->flash_data[addr];
    unsigned char new_data = old_data & byte;

    if ((vlevel > 1) || ((vlevel > 0) && (new_data != byte))) {
        util_message("Flash: Programming $%06x with $%02x ($%02x->$%02x)", addr, byte, old_data, old_data & byte);
    }
    flash040_context->program_byte = byte;
    flash040_context->flash_data[addr] = new_data;
    flash040_context->flash_dirty = 1;

    addr += flash040_context->base_addr;

    if (addr >= flash_prog_end) {
        flash_prog_end = addr + 1;
    }

    if (addr < flash_prog_low) {
        flash_prog_low = addr;
    }

    return (new_data == byte) ? 1 : 0;
}

inline static int flash_write_operation_status(flash040_context_t *flash040_context)
{
    static unsigned int maincpu_clk = 0;

    maincpu_clk += 5;

    return ((flash040_context->program_byte ^ 0x80) & 0x80)   /* DQ7 = inverse of programmed data */
         | ((maincpu_clk & 2) << 5)                           /* DQ6 = toggle bit (2 us) */
         | (1 << 5)                                           /* DQ5 = timeout */
         ;
}

inline static int flash_erase_operation_status(flash040_context_t *flash040_context)
{
    int v;

    /* DQ6 = toggle bit */
    v = flash040_context->program_byte;

    /* toggle the toggle bit(s) */
    /* FIXME better toggle bit II emulation */
    flash040_context->program_byte ^= flash_type.status_toggle_bits;

    /* DQ3 = sector erase timer */
    v |= 0x08;

    return v;
}

/* -------------------------------------------------------------------------- */

static void flash040core_store_internal(flash040_context_t *flash040_context,
                                                 unsigned int addr, unsigned char byte)
{
    switch (flash040_context->flash_state) {
        case FLASH040_STATE_READ:
            if (flash_magic_1(addr) && (byte == 0xaa)) {
                flash040_context->flash_state = FLASH040_STATE_MAGIC_1;
            }
            break;

        case FLASH040_STATE_MAGIC_1:
            if (flash_magic_2(addr) && (byte == 0x55)) {
                flash040_context->flash_state = FLASH040_STATE_MAGIC_2;
            } else {
                flash040_context->flash_state = flash040_context->flash_base_state;
            }
            break;

        case FLASH040_STATE_MAGIC_2:
            if (flash_magic_1(addr)) {
                switch (byte) {
                    case 0x90:
                        flash040_context->flash_state = FLASH040_STATE_AUTOSELECT;
                        flash040_context->flash_base_state = FLASH040_STATE_AUTOSELECT;
                        break;
                    case 0xf0:
                        flash040_context->flash_state = FLASH040_STATE_READ;
                        flash040_context->flash_base_state = FLASH040_STATE_READ;
                        break;
                    case 0xa0:
                        flash040_context->flash_state = FLASH040_STATE_BYTE_PROGRAM;
                        break;
                    case 0x80:
                        flash040_context->flash_state = FLASH040_STATE_ERASE_MAGIC_1;
                        break;
                    default:
                        flash040_context->flash_state = flash040_context->flash_base_state;
                        break;
                }
            } else {
                flash040_context->flash_state = flash040_context->flash_base_state;
            }
            break;

        case FLASH040_STATE_BYTE_PROGRAM:
            if (flash_program_byte(flash040_context, addr, byte)) {
                /* The byte program time is short enough to ignore */
                flash040_context->flash_state = flash040_context->flash_base_state;
            } else {
                flash040_context->flash_state = FLASH040_STATE_BYTE_PROGRAM_ERROR;
                if (vlevel > 0) {
                    util_warning("Flash: Programming error!");
                }
            }
            break;

        case FLASH040_STATE_ERASE_MAGIC_1:
            if (flash_magic_1(addr) && (byte == 0xaa)) {
                flash040_context->flash_state = FLASH040_STATE_ERASE_MAGIC_2;
            } else {
                flash040_context->flash_state = flash040_context->flash_base_state;
            }
            break;

        case FLASH040_STATE_ERASE_MAGIC_2:
            if (flash_magic_2(addr) && (byte == 0x55)) {
                flash040_context->flash_state = FLASH040_STATE_ERASE_SELECT;
            } else {
                flash040_context->flash_state = flash040_context->flash_base_state;
            }
            break;

        case FLASH040_STATE_ERASE_SELECT:
            if (flash_magic_1(addr) && (byte == 0x10)) {
                flash040_context->flash_state = FLASH040_STATE_CHIP_ERASE;
                flash040_context->program_byte = 0;
                flash_erase_chip(flash040_context);
                flash040_context->flash_state = flash040_context->flash_base_state;
            } else if (byte == 0x30) {
                flash040_context->flash_state = FLASH040_STATE_SECTOR_ERASE;
                flash040_context->program_byte = 0;
                flash_erase_sector(flash040_context, addr);
                flash040_context->flash_state = flash040_context->flash_base_state;
            } else {
                flash040_context->flash_state = flash040_context->flash_base_state;
            }
            break;

        case FLASH040_STATE_BYTE_PROGRAM_ERROR:
        case FLASH040_STATE_AUTOSELECT:
            if (flash_magic_1(addr) && (byte == 0xaa)) {
                flash040_context->flash_state = FLASH040_STATE_MAGIC_1;
            }
            if (byte == 0xf0) {
                flash040_context->flash_state = FLASH040_STATE_READ;
                flash040_context->flash_base_state = FLASH040_STATE_READ;
            }
            break;

        case FLASH040_STATE_SECTOR_ERASE:
        case FLASH040_STATE_CHIP_ERASE:
        default:
            break;
    }
}

/* -------------------------------------------------------------------------- */

void flashcore_store(unsigned int addr, unsigned char byte)
{
    unsigned int chip = (addr >> 22) & 3;

    addr &= 0x3fffff;

    flash040core_store_internal(&af_chip[chip], addr, byte);
}


unsigned char flashcore_read(unsigned int addr)
{
    unsigned char value;
    unsigned int chip = (addr >> 22) & 3;
    flash040_context_t *flash040_context = &af_chip[chip];

    addr &= 0x3fffff;

    switch (flash040_context->flash_state) {
        case FLASH040_STATE_AUTOSELECT:
            switch (addr & 0xff) {
                case 0x00:
                    value = flash_type.manufacturer_ID;
                    break;
                case 0x01:
                    value = flash_type.device_ID;
                    break;
                case 0x02:
                    value = 0;
                    break;
                default:
                    value = flash040_context->flash_data[addr];
                    break;
            }
            break;

        case FLASH040_STATE_BYTE_PROGRAM_ERROR:
            value = flash_write_operation_status(flash040_context);
            break;

        case FLASH040_STATE_CHIP_ERASE:
            value = flash_erase_operation_status(flash040_context);
            break;

        default:
            /* The state doesn't reset if a read occurs during a command sequence */
            /* fall through */
        case FLASH040_STATE_READ:
            value = flash040_context->flash_data[addr];
            break;
    }

    return value;
}

static void flash040core_init(struct flash040_context_s *flash040_context, unsigned char *data, unsigned int base_addr)
{
    flash040_context->flash_data = data;
    flash040_context->flash_state = FLASH040_STATE_READ;
    flash040_context->flash_base_state = FLASH040_STATE_READ;
    flash040_context->program_byte = 0;
    flash040_context->flash_dirty = 0;
    flash040_context->base_addr = base_addr;
}

int flashcore_init(int vlevel_in)
{
    unsigned int i;

    vlevel = vlevel_in;

    flash_data = malloc(16 * 1024 * 1024);

    if (flash_data == NULL) {
        util_error("failed to malloc 16MB!");
        return -1;
    }

    flashcore_clear();

    for (i = 0; i < 4; ++i) {
        unsigned int addr = i * 4 * 1024 * 1024;

        flash040core_init(&af_chip[i], &flash_data[addr], addr);
    }

    flash_prog_low = 16 * 1024 * 1024 + 2;
    flash_prog_end = 0;

    return 0;
}

void flashcore_shutdown(void)
{
    free(flash_data);
    flash_data = NULL;
}


void flashcore_clear(void)
{
    memset(flash_data, 0xff, 16 * 1024 * 1024);
}

