#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "aftt.h"
#include "cart.h"
#include "config.h"
#include "globals.h"
#include "misc.h"
#include "util.h"

static void erase_range(int offset, int length)
{
    fprintf(stderr, "Erasing range $%06x-$%06x...\n", offset, offset + length - 1);
    memset(&flash_data[offset], 0xff, length);
}

int erase_afcrt_range(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    int length;

    if (parse_offset(argv[0], &offset, &offset_end, &romh_too) < 0) {
        return -1;
    }

    if (offset_end < 0) {
        /* single address given, erase the sector it lands on */
        offset &= 0xff0000;
        offset_end = offset + 0x010000 - 1;
    }

    length = (offset_end + 1 - offset);

    erase_range(offset, length);

    if (romh_too) {
        offset += 0x800000;
        erase_range(offset, length);
    }

    return 0;
}

int erase_afusb_range(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    size_t datasize;

    if (parse_offset(argv[0], &offset, &offset_end, &romh_too) < 0) {
        return -1;
    }

    if (offset_end < 0) {
        /* single address given, erase the sector it lands on */
        offset &= 0xff0000;
        offset_end = offset + 0x010000 - 1;
    }

    datasize = (offset_end + 1 - offset);

    /* erase the range on the flash image first, in case this is called from the GUI */
    erase_range(offset, datasize);

    if (romh_too) {
        erase_range(offset + 0x800000, datasize);
    }

    return aftt_usb_send(flash_data, datasize, offset, romh_too, 0, 1);
}

static void program_range(unsigned char *b, int num_bytes, int offset, int offset_end)
{
    int i, j;

    fprintf(stderr, "Programming range $%06x-$%06x...\n", offset, offset_end);

    for (i = offset, j = 0; i <= offset_end; ++i, ++j) {
        if (j == num_bytes) {
            j = 0;
        }

        flash_data[i] = b[j];
    }
}

static int program_range_parse_params(int argc, char** argv, int *offset_out, int *offset_end_out, int *romh_too_out, unsigned char *b)
{
    int num_bytes;
    int i, j, t = 0;
    const char *e;

    if (argc < 2) {
        return -1;
    }

    if (parse_offset(argv[0], offset_out, offset_end_out, romh_too_out) < 0) {
        return -1;
    }

    for (j = 0, i = 1; i < argc; ++i, ++j) {
        if (parse_hex(argv[i], &t, 2, &e) < 0) {
            return -1;
        }

        if ((t < 0) || (*e != '\0')) {
            fprintf(stderr, "Error - invalid byte '%s'!\n", argv[i]);
            return -1;
        }

        b[j] = (unsigned char)t;
    }

    num_bytes = j;

    if (*offset_end_out < 0) {
        *offset_end_out = *offset_out + num_bytes - 1;
    }

    if ((*offset_end_out > 0xffffff) || (*romh_too_out && (*offset_end_out > 0x7fffff))) {
        fprintf(stderr, "Error - too many bytes!\n");
        return -1;
    }

    return num_bytes;
}

int program_afusb_range(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    int num_bytes = -1;
    unsigned char b[256];

    num_bytes = program_range_parse_params(argc, argv, &offset, &offset_end, &romh_too, b);

    if (num_bytes < 1) {
        return -1;
    }

    program_range(b, num_bytes, offset, offset_end);

    if (romh_too) {
        program_range(b, num_bytes, offset + 0x800000, offset_end + 0x800000);
    }

    return aftt_usb_send(flash_data, offset_end - offset + 1, offset, romh_too, 0, 0);
}

int program_afcrt_range(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    int num_bytes = -1;
    unsigned char b[256];

    num_bytes = program_range_parse_params(argc, argv, &offset, &offset_end, &romh_too, b);

    if (num_bytes < 1) {
        return -1;
    }

    program_range(b, num_bytes, offset, offset_end);

    if (romh_too) {
        program_range(b, num_bytes, offset + 0x800000, offset_end + 0x800000);
    }

    return 0;
}

static int isempty_range(int offset, int length)
{
    int i;

    fprintf(stderr, "Checking range $%06x-$%06x...\n", offset, offset + length - 1);

    for (i = 0; i < length; ++i) {
        if (flash_data[offset + i] != 0xff) {
            fprintf(stderr, "Not empty, $%06x has $%02x.\n", offset + i, flash_data[offset + i]);
            return 0;
        }
    }
    return 1;
}

int isempty_afcrt_range(int argc, char** argv)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    int isempty;
    int length;

    if (parse_offset(argv[0], &offset, &offset_end, &romh_too) < 0) {
        return -1;
    }

    if (offset_end < 0) {
        /* single address given, check the sector it lands on */
        offset &= 0xff0000;
        offset_end = offset + 0x010000 - 1;
    }

    length = (offset_end + 1 - offset);

    isempty = isempty_range(offset, length);

    if (isempty && romh_too) {
        offset += 0x800000;
        isempty = isempty_range(offset, length);
    }

    if (isempty) {
        fprintf(stderr, "Range is empty.\n");
    }

    return isempty;
}
