#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "afusb.h"
#include "cart.h"
#include "config.h"
#include "convert.h"
#include "flashcore.h"
#include "dump.h"
#include "globals.h"
#include "misc.h"
#include "ui.h"
#include "util.h"

/* ------------------------------------------------------------------------- */
/* ui.h globals */

int verbosity = 2;
const char *outafcrt_name = NULL;

/* ------------------------------------------------------------------------- */

static int inject_prg_to_afcrt(int argc, char** argv)
{
    return inject_file_to_crt_or_usb(argv[0], AFTT_TYPE_PRG, 0);
}

static int inject_d64_to_afcrt(int argc, char** argv)
{
    return inject_file_to_crt_or_usb(argv[0], AFTT_TYPE_D64, 0);
}

static int inject_prg_to_afusb(int argc, char** argv)
{
    return inject_file_to_crt_or_usb(argv[0], AFTT_TYPE_PRG, 1);
}

static int inject_d64_to_afusb(int argc, char** argv)
{
    return inject_file_to_crt_or_usb(argv[0], AFTT_TYPE_D64, 1);
}

static int extract_prg_from_afcrt(int argc, char** argv)
{
    return extract_file_from_crt_or_usb(argv[0], AFTT_TYPE_PRG, 0);
}

static int extract_d64_from_afcrt(int argc, char** argv)
{
    return extract_file_from_crt_or_usb(argv[0], AFTT_TYPE_D64, 0);
}

static int extract_prg_from_afusb(int argc, char** argv)
{
    return extract_file_from_crt_or_usb(argv[0], AFTT_TYPE_PRG, 1);
}

static int extract_d64_from_afusb(int argc, char** argv)
{
    return extract_file_from_crt_or_usb(argv[0], AFTT_TYPE_D64, 1);
}

static int dummy_handle(int argc, char **argv)
{
    int i;

    fprintf(stderr, "got %i params: ", argc);

    for (i = 0; i < argc; ++i) {
        fprintf(stderr, "%s ", argv[i]);
    }

    fprintf(stderr, "\n");

    return 0;
}

#define CMD_NEED_FLASHCORE  (1 << 0)
#define CMD_NEED_AFUSB      (1 << 1)
#define CMD_NEED_LOAD_AFCRT (1 << 2)
#define CMD_NEED_SAVE_AFCRT (1 << 3)

#define CMD_LIST_SEPARATOR(x) { "-", "", x, 0, 0, 0, dummy_handle }

typedef struct tool_commands_s {
    const char *name;
    const char *paramtext;
    const char *helptext;
    int min_argc;
    int max_argc;
    int need_inits;
    int (*handle)(int argc, char **argv);
} tool_commands_t;

static const tool_commands_t tool_commands[] = {

  CMD_LIST_SEPARATOR("Alien Flash .crt - injecting/extracting"),

  { "icrt", "AFCRT INCART [OFFSET]",
    "Inject a .crt file to an Alien Flash .crt file and save the results.",
    2, 3, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT | CMD_NEED_SAVE_AFCRT,
    inject_crt_to_afcrt
  },
  { "id64", "AFCRT IND64",
    "Inject a .d64 image (adding the header) to an Alien Flash .crt and save the results.\n"
    "Destination address is $040000.",
    2, 2, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT | CMD_NEED_SAVE_AFCRT,
    inject_d64_to_afcrt
  },
  { "iprg", "AFCRT INPRG",
    "Inject a .prg file (adding the header) to an Alien Flash .crt and save the results.\n"
    "Destination address is $010000.",
    2, 2, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT | CMD_NEED_SAVE_AFCRT,
    inject_prg_to_afcrt
  },
  { "ibin", "[INAFCRT] INBIN OFFSET",
    "Inject a binary file to an Alien Flash .crt (if given)\n"
    "and save the results to a new Alien Flash .crt file.",
    2, 3, CMD_NEED_FLASHCORE | CMD_NEED_SAVE_AFCRT,
    inject_bin_to_afcrt
  },
  { "ecrt", "INAFCRT OUTCART CRTTYPE [OFFSET]",
    "Extract a .crt file from an Alien Flash .crt.",
    3, 4, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT,
    extract_crt_from_afcrt
  },
  { "ed64", "INAFCRT OUTD64",
    "Extract a .d64 image from an Alien Flash .crt.\n"
    "Source address is $040100.",
    2, 2, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT,
    extract_d64_from_afcrt
  },
  { "eprg", "INAFCRT OUTPRG",
    "Extract a .prg file from an Alien Flash .crt.\n"
    "Source address is $010100.",
    2, 2, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT,
    extract_prg_from_afcrt
  },
  { "aftt2af", "INAFCRT [INAFTT]*",
    "Apply AFTT file(s) on an Alien Flash .crt and save the results.",
    1, 64, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT,
    aftt_to_crt
  },

  CMD_LIST_SEPARATOR("Alien Flash .crt - misc. operations"),

  { "af2bin", "INAFCRT OUTBIN [OFFSET]",
    "Converts an Alien Flash .crt file to a binary file.\n"
    "OFFSET must be a range, default range is 'af' (all Flash).",
    2, 3, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT,
    afcrt_to_bin
  },
  { "erasecrt", "INAFCRT OFFSET",
    "Erase a range of an Alien Flash .crt file and save the results.\n"
    "A single ADDR implies the whole sector, use the ADDR-ADDR\n"
    "format with the same ADDR twice for erasing a single byte.",
    2, 2, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT | CMD_NEED_SAVE_AFCRT,
    erase_afcrt_range
  },
  { "progcrt", "INAFCRT OFFSET BYTE [BYTE]*",
    "Program the BYTE(s) to an Alien Flash .crt file and save the results.\n"
    "If a range is given, the BYTE(s) is/are repeated to fill it.",
    3, 255+3, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT | CMD_NEED_SAVE_AFCRT,
    program_afcrt_range
  },
  { "emptycrt", "INAFCRT OFFSET",
    "Check if the range OFFSET of the Alien Flash .crt is empty.\n"
    "A single ADDR implies the whole sector, use the ADDR-ADDR\n"
    "format with the same ADDR twice for checking a single byte.",
    2, 2, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT,
    isempty_afcrt_range
  },
  { "dumpcrt", "INAFCRT",
    "Dump info on the Alien Flash .crt.",
    1, 1, CMD_NEED_FLASHCORE | CMD_NEED_LOAD_AFCRT,
    dump_afcrt
  },

#ifdef HAVE_AFUSB
  CMD_LIST_SEPARATOR("Alien Flash USB - sending/receiving"),

  { "scrt", "INCART [OFFSET]",
    "Flash a .crt file via Alien Flash USB.",
    1, 2, CMD_NEED_FLASHCORE | CMD_NEED_AFUSB,
    send_crtfile_to_afusb
  },
  { "sd64", "IND64",
    "Flash a .d64 image (adding the header) via Alien Flash USB.\n"
    "Destination address is $040000.",
    1, 1, CMD_NEED_FLASHCORE | CMD_NEED_AFUSB,
    inject_d64_to_afusb
  },
  { "sprg", "INPRG",
    "Flash a .prg file (adding the header) via Alien Flash USB.\n"
    "Destination address is $010000.",
    1, 1, CMD_NEED_FLASHCORE | CMD_NEED_AFUSB,
    inject_prg_to_afusb
  },
  { "rcrt", "OUTCART CRTTYPE [OFFSET]",
    "Retrieve a .crt file via Alien Flash USB.",
    2, 3, CMD_NEED_FLASHCORE | CMD_NEED_AFUSB,
    get_crtfile_from_afusb
  },
  { "rd64", "OUTD64",
    "Retrieve a .d64 image via Alien Flash USB.\n"
    "Source address is $040100.",
    1, 1, CMD_NEED_FLASHCORE | CMD_NEED_AFUSB,
    extract_d64_from_afusb
  },
  { "rprg", "OUTPRG",
    "Retrieve a .prg file via Alien Flash USB.\n"
    "Source address is $010100.",
    1, 1, CMD_NEED_FLASHCORE | CMD_NEED_AFUSB,
    extract_prg_from_afusb
  },
  { "rbin", "OUTBIN OFFSET",
    "Reads data via Alien Flash USB, saves to a binary file.",
    2, 2, CMD_NEED_FLASHCORE | CMD_NEED_AFUSB,
    afusb_to_bin
  },

  CMD_LIST_SEPARATOR("Alien Flash USB - misc. operations"),

  { "erase", "OFFSET",
    "Erase one or more sectors of Alien Flash via USB.",
    1, 1, CMD_NEED_FLASHCORE | CMD_NEED_AFUSB,
    erase_afusb_range
  },
  { "prog", "OFFSET BYTE [BYTE]*",
    "Program the BYTE(s) to Alien Flash via USB. No erasing is performed.\n"
    "If a range is given, the BYTE(s) is/are repeated to fill it.",
    2, 255+2, CMD_NEED_FLASHCORE | CMD_NEED_AFUSB,
    program_afusb_range
  },
  { "nop", "DUMMY",
    "Initialize USB, but don't do any actual operation.\n"
    "The argument DUMMY is ignored, but must exist.",
    1, 1, CMD_NEED_AFUSB,
    dummy_handle
  },
#endif /* HAVE_AFUSB */

  CMD_LIST_SEPARATOR("AFTT file handling"),

  { "sendaftt", "INAFTT [OUTBIN]",
    "Sends the AFTT file via Alien Flash USB, optionally\n"
    "capturing the read bytes to a binary file.",
    1, 2, CMD_NEED_AFUSB,
    aftt_to_afusb
  },
  { "crt2aftt", "INCART OUTAFTT [OFFSET]",
    "Converts a .crt file to an AFTT file.",
    2, 3, CMD_NEED_FLASHCORE,
    crt_to_aftt
  },
  { "d642aftt", "IND64 OUTAFTT",
    "Convert .d64 image to an AFTT file, adding the header.\n"
    "Destination address is $040000.",
    2, 2, 0,
    d64_to_aftt
  },
  { "prg2aftt", "INPRG OUTAFTT",
    "Convert .prg to an AFTT file, adding the header.\n"
    "Destination address is $010000.",
    2, 2, 0,
    prg_to_aftt
  },
  { "aftt2bin", "[INAFTT]* OUTBIN",
    "Apply AFTT file(s) to an empty cartridge and save the\n"
    "programmed range to a raw binary file.",
    1, 64, CMD_NEED_FLASHCORE,
    aftt_to_bin
  },
  { "dumpaftt", "INAFTT [VERBOSE]",
    "Dump the actions of the AFTT file performs, using\n"
    "verbosity level VERBOSE (0..3).\n",
    1, 2, 0,
    dump_aftt
  },

  { NULL, NULL, NULL, 0, 0, 0, dummy_handle }
};

/* ------------------------------------------------------------------------- */

typedef struct tool_args_s {
    const char *name;
    const char *helptext;
} tool_args_t;

static const tool_args_t tool_args[] = {
  { "AFCRT",
    "Alien Flash .crt file."
  },
  { "CART",
    "A cartridge .crt file.\n"
    "The following cart types are supported:\n"
    " * Action Replay (CRT ID 1, default OFFSET ar)\n"
    " * EasyFlash (CRT ID 32, default OFFSET rs1)\n"
    " * Alien Flash (CRT ID 57, default OFFSET af)"
  },
  { "CRTTYPE",
    "Specifies a type of cart.\n"
    "The following types are supported:\n"
    " * ar      Action Replay (CRT ID 1)\n"
    " * ef      EasyFlash (CRT ID 32)\n"
    " * af      Alien Flash (CRT ID 57)"
  },
  { "OFFSET",
    "An address (or range) to the Flash space.\n"
    "The following formats are supported:  (all values in hex)\n"
    " * ADDR        A single address (0 .. ffffff)\n"
    " * ADDR-ADDR   Address range (smaller ADDR first)\n"
    " * rsN         ROM SLOT N (N: 0 .. f)\n"
    " * csIJ        Flash chip I sector J (I: 0 .. 3, J: 0 .. 3f)\n"
    " * ar          Action Replay location (870000 .. 877fff)\n"
    " * at          Alien Flash Tools (0 .. ffff on ROML & ROMH)\n"
    " * af          All (or Alien?) Flash (0 .. ffffff)"
  },
  { "BIN",
    "Raw binary file."
  },
  { "PRG",
    "Normal .prg file."
  },
  { "D64",
    ".d64 disk image file."
  },
  { "AFTT",
    "Alien Flash Terminal Tools file. Meant to be sent to an Alien Flash\n"
    "cartridge via USB using a terminal program, hopefully obsoleted as a\n"
    "format by this tool."
  },

  { "USBDRV",
    "Name of the USB driver: (default is first)\n"
#ifdef HAVE_AFUSB_LINUX
    "\"linux\"\n"
#endif /* HAVE_AFUSB_LINUX */
#ifdef HAVE_AFUSB_WIN32
    "\"win32\"\n"
#endif /* HAVE_AFUSB_WIN32 */
#ifdef HAVE_AFUSB_FTD2XX
    "\"ftd2xx\"\n"
#endif /* HAVE_AFUSB_FTD2XX */
#ifdef HAVE_AFUSB_FTDI
    "\"ftdi\"\n"
#endif /* HAVE_AFUSB_FTDI */
    "\"dummy\""
  },

  { "USBPAR",
    "Depends on the selected USB driver."
#ifdef HAVE_AFUSB_LINUX
    "\nFor \"linux\":\n"
    "  The name of the USB device. (default = " AFUSB_LINUX_DEFAULT_VALUE ")"
#endif /* HAVE_AFUSB_LINUX */
#ifdef HAVE_AFUSB_WIN32
    "\nFor \"win32\":\n"
    "  The number or the virtual COM port. (default = " AFUSB_WIN32_DEFAULT_VALUE ")"
#endif /* HAVE_AFUSB_WIN32 */
#ifdef HAVE_AFUSB_FTDI
    "\nFor \"ftdi\":\n"
    "  A hex value in the range 1 .. ff to set the USB latency;\n"
    "  \"" AFUSB_MAGIC_D2XX_OFF_STRING "\" to set VCP mode;\n"
    "  \"" AFUSB_MAGIC_D2XX_ON_STRING "\" to set D2XX mode."
#endif /* HAVE_AFUSB_FTDI */
    "\nFor \"dummy\":\n"
    "  Dump sent bytes to given filename."
  },

  { NULL, NULL }
};

/* ------------------------------------------------------------------------- */

void usage(int show_helptext)
{
    int i;

    util_message("Usage: aftool [OPTION]* COMMAND ARGS\n\n"
                 "Available COMMANDs:");

    i = 0;
    while (tool_commands[i].name != NULL) {
        if (tool_commands[i].name[0] == '-') {
            if (show_helptext) {
                util_message("\n    %s\n", tool_commands[i].helptext);
            }
            ++i;
            continue;
        }

        util_message("  %s %s",
                    tool_commands[i].name,
                    tool_commands[i].paramtext ? tool_commands[i].paramtext : ""
                  );
        if (show_helptext) {
            util_message("%s\n", tool_commands[i].helptext);
        }
        ++i;
    }

    if (!show_helptext) {
        util_message("\nRun without ARGS for more help on the COMMAND");
    } else {
        util_message("Notes on ARGS:\n");
        i = 0;
        while (tool_args[i].name != NULL) {
            util_message(
                        "  %s\n%s\n",
                        tool_args[i].name,
                        tool_args[i].helptext
                        );
            ++i;
        }
    }

    usage_af_options();
}

static void usage_cmd(int i)
{
    int j = 0;
    int firstarg = 1;

    util_message("aftool %s %s\n\n%s",
            tool_commands[i].name,
            tool_commands[i].paramtext,
            tool_commands[i].helptext
           );

    while (tool_args[j].name != NULL) {
        if (strstr(tool_commands[i].paramtext, tool_args[j].name) != NULL) {
            if (firstarg) {
                util_message("\nNotes on ARGS:\n");
                firstarg = 0;
            }

            util_message(
                        "  %s\n%s\n",
                        tool_args[j].name,
                        tool_args[j].helptext
                        );
        }
        ++j;
    }
}

/* ------------------------------------------------------------------------- */

static int find_command(const char *cmdname)
{
    int i = 0;

    if (*cmdname == '-') {
        return -1;
    }

    while (tool_commands[i].name != NULL) {
        if (strcmp(tool_commands[i].name, cmdname) == 0) {
            break;
        }
        ++i;
    }

    if (tool_commands[i].name != NULL) {
        return i;
    } else {
        return -1;
    }
}

/* ------------------------------------------------------------------------- */

void ui_message(const char *msg)
{
    if (verbosity >= 2) {
        fputs(msg, stdout);
        fputc('\n', stdout);
    }
}

void ui_warning(const char *msg)
{
    if (verbosity > 1) {
        fputs(msg, stderr);
        fputc('\n', stderr);
    }
}

void ui_error(const char *msg)
{
    if (verbosity > 0) {
        fputs(msg, stderr);
        fputc('\n', stderr);
    }
}

/* ------------------------------------------------------------------------- */

int main(int argc, char **argv)
{
    int i = 0;
    int cmd_i;
    int save_i = -1;
    int rc;


    if (argc < 2) {
        usage(0);
        return 0;
    }

    if (parse_options(&argc, argv, &i) < 0) {
        return -1;
    }

    util_message("aftool v" PACKAGE_VERSION " - tools for Alien Flash "
#ifdef HAVE_AFUSB
                 "(built with USB drivers \"" AFUSB_IMPL_NAMES "\")"
#endif
                );

    if (argc < 1) {
        util_error("no COMMAND given!");
        return -1;
    }

    cmd_i = find_command(argv[i]);

    if (cmd_i < 0) {
        util_error("COMMAND '%s' not recognized!", argv[i]);
        return -1;
    }

    argc--;

    if (argc == 0) {
        usage_cmd(cmd_i);
        return 0;
    }

    if (argc < tool_commands[cmd_i].min_argc) {
        util_error("too few arguments for COMMAND '%s'!", argv[i]);
        return -1;
    }

    if (argc > tool_commands[cmd_i].max_argc) {
        util_error("too many arguments for COMMAND '%s'!", argv[i]);
        return -1;
    }

    ++i;

    if (tool_commands[cmd_i].need_inits & CMD_NEED_FLASHCORE) {
        if (flashcore_init(0) < 0) {
            return -1;
        }

        if (tool_commands[cmd_i].need_inits & CMD_NEED_LOAD_AFCRT) {
            if (afcart_load(argv[i]) < 0) {
                flashcore_shutdown();
                return -1;
            }
            save_i = i;
        }

        if (tool_commands[cmd_i].need_inits & CMD_NEED_SAVE_AFCRT) {
            if (outafcrt_name == NULL) {
                /* no OUTAFCRT given, try to use INAFCRT for it */
                if (tool_commands[cmd_i].need_inits & CMD_NEED_LOAD_AFCRT) {
                    outafcrt_name = argv[save_i];
                } else {
                    /* optional INAFCRT; use it as output if given */
                    if (argc > tool_commands[cmd_i].min_argc) {
                        outafcrt_name = argv[i];
                    } else {
                        util_error("No output filename! Supply one with '-o'.");
                        return -1;
                    }
                }
            }
        }

        if (tool_commands[cmd_i].need_inits & CMD_NEED_LOAD_AFCRT) {
            ++i;
            argc--;
        }
    }

    if (tool_commands[cmd_i].need_inits & CMD_NEED_AFUSB) {
        if (afusb_init() < 0) {
            if (tool_commands[cmd_i].need_inits & CMD_NEED_FLASHCORE) {
                flashcore_shutdown();
            }
            return -1;
        }
    }

    rc = tool_commands[cmd_i].handle(argc, &(argv[i]));

    if ((rc >= 0) && (tool_commands[cmd_i].need_inits & CMD_NEED_SAVE_AFCRT)) {
        rc = afcart_save(outafcrt_name, -1, -1);
    }

    util_message("All done%s Shutting down...", (rc < 0) ? ", errors encountered!" : ".");

    if (tool_commands[cmd_i].need_inits & CMD_NEED_FLASHCORE) {
        flashcore_shutdown();
    }

    if (tool_commands[cmd_i].need_inits & CMD_NEED_AFUSB) {
        if (afusb_shutdown() < 0) {
            return -1;
        }
    }

    return rc;
}
