/*
 * ui_fltk.c - afgui FTLK v1.1.x GUI implementation.
 *
 * Based on FLTK examples.
 *
 */

#include <FL/Fl.H>
#include <FL/Fl_Window.H>
#include <FL/Fl_Double_Window.H>
#include <FL/Fl_Box.H>
#include <FL/Fl_Menu_Bar.H>
#include <FL/Fl_Menu_Item.H>
#include <FL/Fl_File_Chooser.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Return_Button.H>
#include <FL/Fl_Group.H>
#include <FL/Fl_Output.H>
#include <FL/Fl_Text_Display.H>
#include <FL/Fl_Text_Buffer.H>
#include <FL/fl_ask.H>

extern "C" {
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "afusb.h"
#include "cart.h"
#include "dump.h"
#include "config.h"
#include "convert.h"
#include "flashcore.h"
#include "globals.h"
#include "ui.h"
#include "util.h"
} /* extern "C" */


/* ------------------------------------------------------------------------- */
/* ui.h globals, unused */

int verbosity = 0;
const char *outafcrt_name = NULL;

/* ------------------------------------------------------------------------- */

#define UI_WIN_W 490
#define UI_WIN_H 480

class af_ui_window : public Fl_Double_Window {
    public:
        af_ui_window(int w, int h, const char *label);
        ~af_ui_window();

        void update_filename(char *fname);
        void update_flashdata(void);
        void update_usbstuff(void);
        void set_usbbusy(bool busy);
        void add_to_log(const char *txt);
        int save_log(const char *file);

    private:
        Fl_Text_Buffer *log_buf_;
        Fl_Text_Display *log_disp_;

        Fl_Box *cartfilename_;

        Fl_Group *group_flashdata_;
        Fl_Box *romslot_[16];
        Fl_Box *slot_ar_;
        Fl_Output *slot_prg_;
        Fl_Output *slot_d64_;

        Fl_Box *box_inject_;
        Fl_Button *button_iprg;
        Fl_Button *button_id64;
        Fl_Button *button_icrt;

        Fl_Box *box_extract_;
        Fl_Button *button_eprg;
        Fl_Button *button_ed64;
        Fl_Button *button_ecrt;

        Fl_Box *usb_header_;

        Fl_Group *group_usb_;
        Fl_Box *usb_status_;
        Fl_Output *usb_param_;
        Fl_Output *usb_driver_;
        Fl_Button *button_usb_conn_;
        Fl_Button *button_usb_disc_;

        Fl_Box *box_send_;
        Fl_Button *button_sprg;
        Fl_Button *button_sd64;
        Fl_Button *button_scrt;

        Fl_Box *box_recv_;
        Fl_Button *button_rprg;
        Fl_Button *button_rd64;
        Fl_Button *button_rcrt;

        int log_lines_;
};

static af_ui_window *ui_win = NULL;

/* ------------------------------------------------------------------------- */

static char filename[1024] = "";
static bool usb_connected = false;

/* ------------------------------------------------------------------------- */

static void ui_load_af_crt_file(char *newfile)
{
    strcpy(filename, "");

    if (afcart_load(newfile) < 0) {
        fl_alert("Error reading from file \'%s\'.", newfile);
        return;
    }

    strcpy(filename, newfile);

    if (ui_win != NULL) {
        ui_win->update_filename(filename);
        ui_win->update_flashdata();
    }
}

static void ui_save_af_crt_file(char *newfile, int offset, int offset_end)
{
    if (afcart_save(newfile, offset, offset_end) < 0) {
        fl_alert("Error saving to file \'%s\'.", newfile);
    } else {
        strcpy(filename, newfile);
    }
}

typedef struct addr_range_data_s {
    const char *name;
    int crt_id;
    const char *range_string;
    const char *type_string;
} addr_range_data_t;

static const addr_range_data_t addr_range_data[] = {
    { "ROM SLOT 0", CARTRIDGE_EASYFLASH, "rs0", "ef" },
    { "ROM SLOT 1", CARTRIDGE_EASYFLASH, "rs1", "ef" },
    { "ROM SLOT 2", CARTRIDGE_EASYFLASH, "rs2", "ef" },
    { "ROM SLOT 3", CARTRIDGE_EASYFLASH, "rs3", "ef" },
    { "ROM SLOT 4", CARTRIDGE_EASYFLASH, "rs4", "ef" },
    { "ROM SLOT 5", CARTRIDGE_EASYFLASH, "rs5", "ef" },
    { "ROM SLOT 6", CARTRIDGE_EASYFLASH, "rs6", "ef" },
    { "ROM SLOT 7", CARTRIDGE_EASYFLASH, "rs7", "ef" },
    { "ROM SLOT 8", CARTRIDGE_EASYFLASH, "rs8", "ef" },
    { "ROM SLOT 9", CARTRIDGE_EASYFLASH, "rs9", "ef" },
    { "ROM SLOT 10", CARTRIDGE_EASYFLASH, "rsa", "ef" },
    { "ROM SLOT 11", CARTRIDGE_EASYFLASH, "rsb", "ef" },
    { "ROM SLOT 12", CARTRIDGE_EASYFLASH, "rsc", "ef" },
    { "ROM SLOT 13", CARTRIDGE_EASYFLASH, "rsd", "ef" },
    { "ROM SLOT 14", CARTRIDGE_EASYFLASH, "rse", "ef" },
    { "ROM SLOT 15", CARTRIDGE_EASYFLASH, "rsf", "ef" },
    { "Alien Flash", CARTRIDGE_ALIEN_FLASH, "af", "af" },
    { "Alien Flash Tools", CARTRIDGE_ALIEN_FLASH, "at", "af" },
    { "Action Replay", CARTRIDGE_ACTION_REPLAY, "ar", "ar" },
    { "PRG", -(AFTT_TYPE_PRG), NULL, NULL },
    { "D64", -(AFTT_TYPE_D64), NULL, NULL },
    { "Custom", -1, NULL, NULL },
    { NULL }
};

static Fl_Menu_Item addr_range_items[] = {
    { "ROM SLOT 0", 0, NULL },
    { "ROM SLOT 1", 0, NULL },
    { "ROM SLOT 2", 0, NULL },
    { "ROM SLOT 3", 0, NULL },
    { "ROM SLOT 4", 0, NULL },
    { "ROM SLOT 5", 0, NULL },
    { "ROM SLOT 6", 0, NULL },
    { "ROM SLOT 7", 0, NULL },
    { "ROM SLOT 8", 0, NULL },
    { "ROM SLOT 9", 0, NULL },
    { "ROM SLOT 10", 0, NULL },
    { "ROM SLOT 11", 0, NULL },
    { "ROM SLOT 12", 0, NULL },
    { "ROM SLOT 13", 0, NULL },
    { "ROM SLOT 14", 0, NULL },
    { "ROM SLOT 15", 0, NULL },
    { "Alien Flash", 0, NULL },
    { "Alien Flash Tools", 0, NULL },
    { "Action Replay", 0, NULL },
    { "PRG", 0, NULL },
    { "D64", 0, NULL },
    { "Custom", 0, NULL },
    { NULL }
};

static int ui_range_dialog(const char *op_text, char *file,
                           char *cart_name, int crt_id,
                           int *offset_out, int *offset_end_out, int *romh_too_out,
                           char **range_out, char **type_out)
{
    switch (crt_id) {
        case CARTRIDGE_ACTION_REPLAY:
        case CARTRIDGE_ALIEN_FLASH:
            /* just use the default address */
            return 0;

        default:
            break;
    }

    /* deactivate unused items */
    for (int i = 0; addr_range_data[i].name != NULL; ++i) {
        if (0
            || ((crt_id >= 0) && (addr_range_data[i].crt_id < -1))
            || ((crt_id < 0) && (addr_range_data[i].crt_id > 0))) {
            addr_range_items[i].hide();
        } else {
            addr_range_items[i].show();
        }
    }

    int w = 180;
    int h = 70;
    int x = 0;
    int y = 0;

    if (cart_name != NULL) {
        h += 20;
    }

    Fl_Window *s_win = new Fl_Window(w, h, op_text);
    Fl_Box *s_text = NULL;
    Fl_Choice *s_choice = NULL;
    Fl_Return_Button *s_ok = NULL;
    Fl_Button *s_cancel = NULL;

    s_win->begin();
    {
        if (cart_name != NULL) {
            s_text = new Fl_Box(FL_NO_BOX, x, y, w, 25, cart_name);
            y += 25;
        } else {
            y += 5;
        }

        s_choice = new Fl_Choice(x + 5, y, w - 5, 30);
        s_choice->menu(addr_range_items);
        y += 35;

        s_ok = new Fl_Return_Button(w - 60 - 65 - 15, y, 60, 25, "OK");

        s_cancel = new Fl_Button(w - 65, y, 60, 25, "Cancel");
        s_cancel->type(FL_NORMAL_BUTTON);
        s_cancel->shortcut(FL_Escape);
    }
    s_win->end();
    s_win->show();

    /* deactivate Fl::grab(), because it is incompatible with Fl::readqueue() */
    Fl_Window* g = Fl::grab();

    /* do an alternative grab to avoid floating menus, if possible */
    if (g) {
        Fl::grab(s_win);
    }

    int r;

    for (;;) {
        Fl_Widget *o = Fl::readqueue();

        if (!o) {
            Fl::wait();
        } else if (o == s_ok) {
            r = 0;
            break;
        } else if (o == s_cancel) {
            r = -1;
            break;
        }
    }

    /* regrab the previous popup menu, if there was one */
    if (g) {
        Fl::grab(g);
    }

    s_win->hide();

    /* get offset */
    int i = s_choice->value();

    /* delete window */
    delete s_win;
    s_win = NULL;

    char *range_string = (char *)(addr_range_data[i].range_string);

    if (range_out) {
        *range_out = range_string;
    }

    char *type_string = (char *)(addr_range_data[i].type_string);

    if (type_out) {
        *type_out = type_string;
    }

    /* TODO handle custom offsets */
    if (parse_offset(range_string, offset_out, offset_end_out, romh_too_out) < 0) {
        fl_alert("Bug in ui_range_dialog!");
        return -1;
    }

    return r;
}

/* ------------------------------------------------------------------------- */

void usb_driver_cb(Fl_Widget *w, void *p)
{
    if (afusb_select_driver((const char *)p) < 0) {
        fl_alert("Problems selecting driver!");
        return;
    }
    ui_win->update_usbstuff();
}

void usb_conn_cb(Fl_Widget *w, void *)
{
    if (afusb_init() < 0) {
        fl_alert("Problems connecting");
        return;
    }
    usb_connected = true;
    ui_win->update_usbstuff();
}

void usb_disconn_cb(Fl_Widget *w, void *)
{
    usb_connected = false;
    if (afusb_shutdown() < 0) {
        fl_alert("Problems disconnecting");
    }
    ui_win->update_usbstuff();
}

void usb_param_cb(Fl_Widget *w, void *p)
{
    char *param_next = (char *)p;

    if (!afusb_driver_has_params()) {
        fl_alert("Driver has no parameters!");
        return;
    }

    if (param_next == NULL) {
        const char *param_old;

        if (afusb_get_device_param(&param_old) < 0) {
            fl_alert("Problems getting old parameter!");
            return;
        }

        /* parameter dialog */
        param_next = (char *)fl_input("Set USB parameter:", param_old);
    }

    if (param_next == NULL) {
        /* no parameter on canceled */
        return;
    }

    afusb_set_device_param(1, &param_next);

    ui_win->update_usbstuff();
}

static bool check_usb_conn(void)
{
    if (usb_connected) {
        return true;
    }

    fl_alert("USB not connected, try connecting first.");
    return false;
}

void sd64_cb(Fl_Widget*, void*)
{
    if (!check_usb_conn()) {
        return;
    }

    ui_win->set_usbbusy(true);
    int rc = inject_file_to_crt_or_usb(NULL, AFTT_TYPE_D64, 1);
    ui_win->set_usbbusy(false);

    if (rc < 0) {
        fl_alert("Problems with USB, disconnecting");
        usb_disconn_cb(NULL, NULL);
    }
}

void sprg_cb(Fl_Widget*, void*)
{
    if (!check_usb_conn()) {
        return;
    }

    ui_win->set_usbbusy(true);
    int rc = inject_file_to_crt_or_usb(NULL, AFTT_TYPE_PRG, 1);
    ui_win->set_usbbusy(false);

    if (rc < 0) {
        fl_alert("Problems with USB, disconnecting");
        usb_disconn_cb(NULL, NULL);
    }
}

void rd64_cb(Fl_Widget*, void*)
{
    if (!check_usb_conn()) {
        return;
    }

    ui_win->set_usbbusy(true);
    int rc = extract_file_from_crt_or_usb(NULL, AFTT_TYPE_D64, 1);
    ui_win->set_usbbusy(false);

    ui_win->update_flashdata();

    if (rc < 0) {
        fl_alert("Problems with USB, disconnecting");
        usb_disconn_cb(NULL, NULL);
    }
}

void rprg_cb(Fl_Widget*, void*)
{
    if (!check_usb_conn()) {
        return;
    }

    ui_win->set_usbbusy(true);
    int rc = extract_file_from_crt_or_usb(NULL, AFTT_TYPE_PRG, 1);
    ui_win->set_usbbusy(false);

    ui_win->update_flashdata();

    if (rc < 0) {
        fl_alert("Problems with USB, disconnecting");
        usb_disconn_cb(NULL, NULL);
    }
}

void scrt_cb(Fl_Widget*, void*)
{
    if (!check_usb_conn()) {
        return;
    }

    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    int crt_id = 0;
    int fake_argc = 2;
    char *fake_argv[2] = { NULL, NULL };

    if (ui_range_dialog("Send", NULL, NULL, crt_id, &offset, &offset_end, &romh_too, &fake_argv[1], NULL) < 0) {
        return;
    }

    /* TODO custom -> use offsets */
    ui_win->set_usbbusy(true);
    int rc = send_crtfile_to_afusb(fake_argc, fake_argv);
    ui_win->set_usbbusy(false);

    if (rc < 0) {
        fl_alert("Problems with USB, disconnecting");
        usb_disconn_cb(NULL, NULL);
    }
}

void rcrt_cb(Fl_Widget*, void*)
{
    if (!check_usb_conn()) {
        return;
    }

    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    int crt_id = 0;
    int fake_argc = 3;
    char *fake_argv[3] = { NULL, NULL, NULL };

    if (ui_range_dialog("Receive", NULL, NULL, crt_id, &offset, &offset_end, &romh_too, &fake_argv[2], &fake_argv[1]) < 0) {
        return;
    }

    /* TODO custom -> use offsets */
    int rc = get_crtfile_from_afusb(fake_argc, fake_argv);

    ui_win->update_flashdata();

    if (rc < 0) {
        fl_alert("Problems with USB, disconnecting");
        usb_disconn_cb(NULL, NULL);
    }
}

/* ------------------------------------------------------------------------- */

void new_cb(Fl_Widget *, void *)
{
    filename[0] = '\0';
    flashcore_clear();
    ui_win->update_flashdata();
}

void open_cb(Fl_Widget*, void*)
{
    char *newfile = fl_file_chooser("Open Alien Flash file", "*.crt", filename);

    if (newfile != NULL) {
        ui_load_af_crt_file(newfile);
    }
}

void saveas_cb(Fl_Widget*, void*)
{
    char *newfile = fl_file_chooser("Save Alien Flash file as?", "*.crt", filename);

    if (newfile != NULL) {
        ui_save_af_crt_file(newfile, -1, -1);
        strcpy(filename, newfile);
    }
}

void save_cb(Fl_Widget*, void*)
{
    if (filename[0] == '\0') {
        /* No filename - get one! */
        saveas_cb(NULL, NULL);
        return;
    } else {
        ui_save_af_crt_file(filename, -1, -1);
    }
}

void savelog_cb(Fl_Widget*, void*)
{
    char *newfile = fl_file_chooser("Save log as?", "*.txt", "log.txt");

    if (newfile == NULL) {
        return;
    }

    if (ui_win->save_log((const char *)newfile) != 0) {
        fl_alert("Problems saving log");
    }
}

void quit_cb(Fl_Widget *w, void *p)
{
    exit(0);
}

/* ------------------------------------------------------------------------- */

void id64_cb(Fl_Widget*, void*)
{
    char *newfile = fl_file_chooser("Open D64 file", "*.d64", NULL);

    if (newfile != NULL) {
        if (inject_file_to_crt_or_usb(newfile, AFTT_TYPE_D64, 0) < 0) {
            fl_alert("Error injecting file!");
        }
    }
    ui_win->update_flashdata();
}

void iprg_cb(Fl_Widget*, void*)
{
    char *newfile = fl_file_chooser("Open PRG file", "*.prg", NULL);

    if (newfile != NULL) {
        if (inject_file_to_crt_or_usb(newfile, AFTT_TYPE_PRG, 0) < 0) {
            fl_alert("Error injecting file!");
        }
    }
    ui_win->update_flashdata();
}

void ed64_cb(Fl_Widget*, void*)
{
    char *newfile;
    char name[17];
    int type = AFTT_TYPE_D64;

    if (!has_aftt_file(type, name)) {
        fl_alert("No D64 found!");
        return;
    }

    newfile = fl_file_chooser("Save D64 as?", "*.d64", NULL);

    if (newfile != NULL) {
        if (extract_file_from_crt_or_usb(newfile, type, 0) < 0) {
            fl_alert("Error extracting file!");
        }
    }
}

void eprg_cb(Fl_Widget*, void*)
{
    char *newfile;
    char name[17];
    int type = AFTT_TYPE_PRG;

    if (!has_aftt_file(type, name)) {
        fl_alert("No PRG found!");
        return;
    }

    newfile = fl_file_chooser("Save PRG as?", "*.prg", NULL);

    if (newfile != NULL) {
        if (extract_file_from_crt_or_usb(newfile, type, 0) < 0) {
            fl_alert("Error extracting file!");
        }
    }
}

void icrt_cb(Fl_Widget*, void*)
{
    int offset = -1;
    int crt_id = -1;
    char *cart_name = NULL;

    char *newfile = fl_file_chooser("Open CRT file", "*.crt", NULL);

    if (newfile == NULL) {
        return;
    }

    if (detect_cart_type(newfile, &offset, &crt_id, (const char **)&cart_name) < 0) {
        fl_alert("Error injecting CRT file '%s' (ID %i)!", newfile, crt_id);
        return;
    }

    if (ui_range_dialog("Inject", newfile, cart_name, crt_id, &offset, NULL, NULL, NULL, NULL) < 0) {
        return;
    }

    if (anycart_load(newfile, &offset, NULL, NULL) < 0) {
        fl_alert("Error injecting file!");
    }
    ui_win->update_flashdata();
}

void ecrt_cb(Fl_Widget*, void*)
{
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    int crt_id = 0;
    int fake_argc = 3;
    char *fake_argv[3];

    if (ui_range_dialog("Extract", NULL, NULL, crt_id, &offset, &offset_end, &romh_too, &fake_argv[2], &fake_argv[1]) < 0) {
        return;
    }

    fake_argv[0] = fl_file_chooser("Save CRT file as?", "*.crt", NULL);

    if (fake_argv[0] == NULL) {
        return;
    }

    /* TODO custom -> use offsets */
    if (extract_crt_from_afcrt(fake_argc, fake_argv) < 0) {
        fl_alert("Error extracting file!");
    }
}

void dump_cb(Fl_Widget*, void*)
{
    dump_afcrt(0, NULL);
}

/* ------------------------------------------------------------------------- */

Fl_Menu_Item menuitems[] = {
    { "&File",                  0, NULL, NULL, FL_SUBMENU },
        { "&New image file",          0, (Fl_Callback *)new_cb },
        { "&Open image file...",      FL_CTRL + 'o', (Fl_Callback *)open_cb },
        { "&Save image file",         FL_CTRL + 's', (Fl_Callback *)save_cb },
        { "Save image file &as...",   FL_CTRL + FL_SHIFT + 's', (Fl_Callback *)saveas_cb, NULL, FL_MENU_DIVIDER },
        { "E&xit",              FL_CTRL + 'q', (Fl_Callback *)quit_cb, NULL },
        { NULL },

    { "&Image",                 0, NULL, NULL, FL_SUBMENU },
        { "&Inject file",       0, NULL, NULL, FL_SUBMENU },
            { "&CRT...",        0, (Fl_Callback *)icrt_cb },
            { "&D64...",        0, (Fl_Callback *)id64_cb },
            { "&PRG...",        0, (Fl_Callback *)iprg_cb },
            { NULL },

        { "&Extract file",      0, NULL, NULL, FL_SUBMENU },
            { "&CRT...",        0, (Fl_Callback *)ecrt_cb },
            { "&D64...",        0, (Fl_Callback *)ed64_cb },
            { "&PRG...",        0, (Fl_Callback *)eprg_cb },
            { NULL },

        { "&Dump info", 0, (Fl_Callback *)dump_cb },

        { NULL },

    { "&USB",                   0, NULL, NULL, FL_SUBMENU },
        { "&Select USB driver", 0, NULL, NULL, FL_SUBMENU },
#ifdef HAVE_AFUSB_WIN32
            { "win32", 0, (Fl_Callback *)usb_driver_cb, (void *)"win32" },
#endif /* HAVE_AFUSB_WIN32 */
#ifdef HAVE_AFUSB_LINUX
            { "linux", 0, (Fl_Callback *)usb_driver_cb, (void *)"linux" },
#endif /* HAVE_AFUSB_LINUX */
#ifdef HAVE_AFUSB_FTDI
            { "ftdi", 0, (Fl_Callback *)usb_driver_cb, (void *)"ftdi" },
#endif /* HAVE_AFUSB_FTDI */
#ifdef HAVE_AFUSB_FTD2XX
            { "ftd2xx", 0, (Fl_Callback *)usb_driver_cb, (void *)"ftd2xx" },
#endif /* HAVE_AFUSB_FTD2XX */
            { "dummy", 0, (Fl_Callback *)usb_driver_cb, (void *)"dummy" },
            { NULL },

        { "&Set parameter...", 0, (Fl_Callback *)usb_param_cb, NULL, FL_MENU_DIVIDER },

        { "&Connect",           0, (Fl_Callback *)usb_conn_cb },
        { "&Disconnect",        0, (Fl_Callback *)usb_disconn_cb, NULL, FL_MENU_DIVIDER },

        { "&Send from image",   0, NULL, NULL, FL_SUBMENU },
            { "&CRT...",        0, (Fl_Callback *)scrt_cb },
            { "&D64",           0, (Fl_Callback *)sd64_cb },
            { "&PRG",           0, (Fl_Callback *)sprg_cb },
            { NULL },

        { "&Receive to image",  0, NULL, NULL, FL_SUBMENU },
            { "&CRT...",        0, (Fl_Callback *)rcrt_cb },
            { "&D64",           0, (Fl_Callback *)rd64_cb },
            { "&PRG",           0, (Fl_Callback *)rprg_cb },
            { NULL },

        { NULL },

    { "&Misc",                  0, NULL, NULL, FL_SUBMENU },
        { "Save log &as...",    0, (Fl_Callback *)savelog_cb },
        { NULL },

    { NULL }
};

af_ui_window::af_ui_window(int w, int h, const char* label) : Fl_Double_Window(w, h, label)
{
    log_lines_ = 0;

    int x = 5;
    int y = 0;

    int oper_box_y = 240;

    {
        Fl_Menu_Bar *m = new Fl_Menu_Bar(0, 0, w, 30);
        m->copy(menuitems);
        y += 30;
    }

    /* bottom: log */
    {
        int yo = y + 330 + 25;

        log_buf_ = new Fl_Text_Buffer();
        log_disp_ = new Fl_Text_Display(x, yo, w - x - 5, h - yo, "Log");
        log_disp_->align(FL_ALIGN_TOP_LEFT);
        log_disp_->buffer(log_buf_);
        log_disp_->hide_cursor();
        log_disp_->textfont(FL_COURIER);
        log_disp_->textsize(12);
    }

    /* left side: cart image */
    int yt = y;
    int wt = 128 + 20;
    int xt = x;
    {
        y += 5;
        cartfilename_ = new Fl_Box(FL_NO_BOX, x, y, wt * 2, 20, "...");
        update_filename(filename);
        y += 25;
    }

    {
        int ht = 320;

        group_flashdata_ = new Fl_Group(x, y, wt * 2 + 20, ht);
        {
            y = yt + 30;

            slot_ar_ = new Fl_Box(FL_BORDER_BOX, x, y, wt, ht / 16, "Action Replay");
            y += (ht / 16) + 5;

            slot_prg_ = new Fl_Output(x, y, wt, 25);

            y += (ht / 16) + 10;

            slot_d64_ = new Fl_Output(x, y, wt, 25);

            y += (ht / 16) + 20;

            ht = 45;
            y = oper_box_y;
            int wb = (wt - 20) / 3;

            box_inject_ = new Fl_Box(FL_BORDER_BOX, x, y, wt, ht, "Inject");
            box_inject_->align(FL_ALIGN_INSIDE | FL_ALIGN_TOP);
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "CRT");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)icrt_cb, 0);
                button_icrt = b;
            }
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "PRG");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)iprg_cb, 0);
                button_iprg = b;
            }
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "D64");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)id64_cb, 0);
                button_id64 = b;
            }
            y += ht + 20;

            x = xt;
            box_inject_ = new Fl_Box(FL_BORDER_BOX, x, y, wt, ht, "Extract");
            box_inject_->align(FL_ALIGN_INSIDE | FL_ALIGN_TOP);
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "CRT");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)ecrt_cb, 0);
                button_icrt = b;
            }
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "PRG");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)eprg_cb, 0);
                button_iprg = b;
            }
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "D64");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)ed64_cb, 0);
                button_id64 = b;
            }

            /* middle: ROM SLOT table */
            wt -= 20;
            x = xt + wt + 35;
            y = yt + 30;
            ht = 320;

            for (int i = 15; i >= 0; --i) {
                char ltxt[20];
                sprintf(ltxt, "ROM SLOT %i", i);

                romslot_[i] = new Fl_Box(FL_BORDER_BOX, x + 5, y, wt, ht / 16, ".");
                romslot_[i]->copy_label(ltxt);
                y += (ht / 16);
            }

            update_flashdata();

        }
        group_flashdata_->end();
    }

    /* right side: cart image */
    y = yt;
    wt = 158;
    xt = 320;
    x = xt;
    {
        y += 5;
        usb_header_ = new Fl_Box(FL_NO_BOX, x, y, wt, 20, "Alien Flash USB");
        y += 25;
    }

    {
        int ht = 320;

        group_usb_ = new Fl_Group(x, y, wt + 20, ht);
        {
            usb_status_ = new Fl_Box(FL_BORDER_BOX, x, y, wt, ht / 16, "...");
            y += (ht / 16) + 5;

            usb_driver_ = new Fl_Output(x + 50, y, wt - 50, 25, "Driver:");

            y += (ht / 16) + 10;

            usb_param_ = new Fl_Output(x + 50, y, wt - 50, 25, "Param:");

            y += (ht / 16) + 10;

            {
                Fl_Button *b = new Fl_Button(x, y, wt, 20, "Connect");
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)usb_conn_cb, 0);
                button_usb_conn_ = b;
            }

            y += 20;

            {
                Fl_Button *b = new Fl_Button(x, y, wt, 20, "Disconnect");
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)usb_disconn_cb, 0);
                button_usb_disc_ = b;
            }

            x = xt;
            ht = 45;
            y = oper_box_y;
            int wb = (wt - 20) / 3;

            box_send_ = new Fl_Box(FL_BORDER_BOX, x, y, wt, ht, "Send");
            box_send_->align(FL_ALIGN_INSIDE | FL_ALIGN_TOP);
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "CRT");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)scrt_cb, 0);
                button_scrt = b;
            }
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "PRG");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)sprg_cb, 0);
                button_sprg = b;
            }
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "D64");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)sd64_cb, 0);
                button_sd64 = b;
            }
            y += ht + 20;

            x = xt;
            box_recv_ = new Fl_Box(FL_BORDER_BOX, x, y, wt, ht, "Receive");
            box_recv_->align(FL_ALIGN_INSIDE | FL_ALIGN_TOP);
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "CRT");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)rcrt_cb, 0);
                button_icrt = b;
            }
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "PRG");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)rprg_cb, 0);
                button_rprg = b;
            }
            {
                Fl_Button *b = new Fl_Button(x + 5, y + 20, wb, 20, "D64");
                x += wb + 5;
                b->type(FL_NORMAL_BUTTON);
                b->callback((Fl_Callback *)rd64_cb, 0);
                button_rd64 = b;
            }

        }
        group_usb_->end();
        update_usbstuff();
    }
}

af_ui_window::~af_ui_window()
{
}

void af_ui_window::add_to_log(const char *txt)
{
    int i = 0;

    for (i = 0; (i < 1024) && (txt[i] != '\0'); ++i) {
        if (txt[i] == '\n') {
            log_lines_++;
        }
    }

    log_lines_++;

    log_disp_->insert(txt);
    log_disp_->insert("\n");

    if (log_lines_ > 3) {
        log_disp_->scroll(log_lines_ - 3, 0);
    }
}

int af_ui_window::save_log(const char *filename)
{
    return log_buf_->savefile(filename);
}

void af_ui_window::update_filename(char *fname)
{
    cartfilename_->label((*fname != '\0') ? fl_filename_name(fname) : "(no file)");
}

static Fl_Color afcolorromslot(int slot)
{
    int addr = slot * 0x080000;
    int i;

    if (detect_bootvec(addr + 0x800000)) {
        return FL_GREEN;
    }

    if (detect_cbm80(addr)) {
        return FL_BLUE;
    }

    for (i = 0; i < 0x080000; ++i) {
        if (flash_data[addr + i] != 0xff) {
            return FL_YELLOW;
        }
    }

    addr += 0x800000;

    for (i = 0; i < 0x080000; ++i) {
        if (flash_data[addr + i] != 0xff) {
            return FL_YELLOW;
        }
    }

    return FL_INACTIVE_COLOR;
}

void af_ui_window::update_flashdata(void)
{
    for (int i = 0; i < 16; ++i) {
        romslot_[i]->color(afcolorromslot(i));
        romslot_[i]->redraw_label();
    }

    {
        char name[17] = "<no D64 file>";
        int j;

        j = has_aftt_file(AFTT_TYPE_D64, name);

        slot_d64_->color(j ? FL_GREEN : FL_INACTIVE_COLOR);
        slot_d64_->value(name);
        slot_d64_->redraw();
    }

    {
        char name[17] = "<no PRG file>";
        int j;

        j = has_aftt_file(AFTT_TYPE_PRG, name);

        slot_prg_->color(j ? FL_GREEN : FL_INACTIVE_COLOR);
        slot_prg_->value(name);
        slot_prg_->redraw();
    }

    slot_ar_->color(detect_ar(-1) ? FL_GREEN : FL_INACTIVE_COLOR);
    slot_ar_->redraw();
}

void af_ui_window::update_usbstuff(void)
{
    usb_connected = afusb_is_connected() ? true : false;

    usb_status_->label(usb_connected ? "Connected" : "Disconnected");
    usb_status_->color(usb_connected ? FL_GREEN : FL_INACTIVE_COLOR);

    usb_driver_->value(afusb_get_driver_name());

    {
        const char *s = "<no params>";

        if (afusb_driver_has_params()) {
            afusb_get_device_param(&s);
        }

        usb_param_->value(s);
    }

    if (usb_connected) {
        /* TODO enable/disable controls? */
    }
}

void af_ui_window::set_usbbusy(bool busy)
{
    usb_status_->label(busy ? "Busy" : "Connected");
    usb_status_->color(busy ? FL_RED : FL_GREEN);

    if (busy) {
        /* TODO force redraw? */
    }
}

/* ------------------------------------------------------------------------- */

static void ui_abort(const char *, ...)
{
    exit(0);
}

static void afgui_shutdown(void)
{
    fprintf(stdout, "afgui shutting down.\n");

    delete ui_win;
    ui_win = NULL;

    afusb_shutdown();
    flashcore_shutdown();
}

/* ------------------------------------------------------------------------- */

extern "C" {

void ui_message(const char *msg)
{
    if (ui_win != NULL) {
        ui_win->add_to_log(msg);
    } else {
        fputs(msg, stdout);
        fputc('\n', stdout);
    }
}

void ui_warning(const char *msg)
{
    if (ui_win != NULL) {
        ui_win->add_to_log(msg);
    } else {
        fputs(msg, stderr);
        fputc('\n', stderr);
    }
}

void ui_error(const char *msg)
{
    if (ui_win != NULL) {
        ui_win->add_to_log(msg);
    } else {
        fputs(msg, stderr);
        fputc('\n', stderr);
    }
}

void usage(int show_helptext)
{
    util_message("Usage: afgui [AFCRT]");
    usage_af_options();
}

} /* extern "C" */

/* ------------------------------------------------------------------------- */

int main(int argc, char **argv)
{
    int i = 0;

    fprintf(stdout, "afgui v" VERSION "\n");

    if (parse_options(&argc, argv, &i) < 0) {
        return -1;
    }

    atexit(afgui_shutdown);

    if (flashcore_init(0) < 0) {
        return -1;
    }

    Fl::set_abort(ui_abort);

    ui_win = new af_ui_window(UI_WIN_W, UI_WIN_H, "afgui v" VERSION);

    Fl::visual(FL_DOUBLE|FL_INDEX);
    ui_win->show();

    ui_message("afgui v" VERSION " - GUI for Alien Flash\n(built with USB drivers \"" AFUSB_IMPL_NAMES "\")\nRead the README file(s) for help.");

    if (argc > 0) {
        ui_load_af_crt_file(argv[i]);
    }

    return Fl::run();
}
