#include <ctype.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"

#ifndef IS_WINDOWS
#include <unistd.h>
#else
#include <windef.h>
#include <winbase.h>
#endif

#include "aftt.h"
#include "afusb.h"
#include "cart.h"
#include "globals.h"
#include "ui.h"
#include "util.h"

/* ------------------------------------------------------------------------- */

#define MAX_MSG_LEN (1024 * 2)

static char msgbuf[MAX_MSG_LEN] = "";

void util_message(const char* format, ...)
{
    va_list ap;
    int len;

    va_start(ap, format);
    len = vsprintf(msgbuf, format, ap);
    va_end(ap);

    ui_message(msgbuf);
}

void util_warning(const char* format, ...)
{
    va_list ap;
    int len;

    len = sprintf(msgbuf, "Warning - ");

    va_start(ap, format);
    len = vsprintf(&msgbuf[len], format, ap);
    va_end(ap);

    ui_warning(msgbuf);
}

void util_error(const char* format, ...)
{
    va_list ap;
    int len;

    len = sprintf(msgbuf, "Error - ");

    va_start(ap, format);
    len = vsprintf(&msgbuf[len], format, ap);
    va_end(ap);

    ui_error(msgbuf);
}

/* ------------------------------------------------------------------------- */

int parse_hex(const char *s, int *v_out, int max_i, const char **e)
{
    int rc = 0;
    int i = 0;
    unsigned int v = 0;
    unsigned char b;

    while ((i < max_i) && (s[i] != '\0')) {
        b = tolower(s[i]);

        if (b == '-') {
            break;
        }

        if (b < '0') {
            rc = -1;
            break;
        }

        b -= '0';

        if (b > 9) {
            b -= 'a' - ('9' + 1);
        }

        if (b > 15) {
            rc = -1;
            break;
        }

        v <<= 4;
        v |= b;

        ++i;
    }

    *v_out = v;

    if (e != NULL) {
        *e = s + i;
    }

    return rc;
}

int parse_offset(const char *s, int *offset_out, int *offset_end_out, int *romh_too_out)
{
    int rc = -1;
    int offset = -1;
    int offset_end = -1;
    int romh_too = 0;
    const char *e = NULL;

    /* handle special OFFSET types */
    if ((s[0] == 'r') && (s[1] == 's') && (s[2] != '\0')) {
        /* rsN: ROM SLOT */
        rc = parse_hex(&s[2], &offset, 1, &e);

        if (rc || (*e != '\0')) {
            util_error("invalid ROM SLOT string '%s'", &s[2]);
            return -1;
        }

        offset *= 0x080000;
        offset_end = offset + 0x080000 - 1;
        /* ROM SLOT implies the ROMH too */
        romh_too = 1;
    } else if ((s[0] == 'c') && (s[1] == 's') && (s[2] != '\0')) {
        /* csIJ: chip/sector */
        int v = -1;

        rc = parse_hex(&s[2], &v, 1, &e);

        if (rc || (*e == '\0') || (v < 0) || (v > 3)) {
            util_error("invalid chip/sector string '%s'", &s[2]);
            return -1;
        }

        offset = v * 0x400000;

        rc = parse_hex(&s[3], &v, 2, &e);

        if (rc || (*e != '\0') || (v < 0) || (v > 0x3f)) {
            util_error("invalid chip/sector string '%s'", &s[2]);
            return -1;
        }

        offset += v * 0x010000;
        offset_end = offset + 0x010000 - 1;
    } else if ((s[0] == 'a') && (s[1] == 'r') && (s[2] == '\0')) {
        /* ar: Action Replay location */
        rc = 0;
        offset = 0x870000;
        offset_end = offset + 0x08000 - 1;
    } else if ((s[0] == 'a') && (s[1] == 'f') && (s[2] == '\0')) {
        /* af: all flash */
        rc = 0;
        offset = 0;
        offset_end = 0xffffff;
    } else if ((s[0] == 'a') && (s[1] == 't') && (s[2] == '\0')) {
        /* at: Alien Flash Tools */
        rc = 0;
        offset = 0;
        offset_end = 0x00ffff;
        romh_too = 1;
    } else {
        /* handle ADDR and ADDR-ADDR  */
        rc = parse_hex(s, &offset, 6, &e);

        if (rc || ((*e != '-') && (*e != '\0'))) {
            util_error("invalid ADDR string '%s'", s);
            return -1;
        }

        /* handle ADDR-ADDR  */
        if (*e == '-') {
            const char *p = e + 1;

            rc = parse_hex(p, &offset_end, 6, &e);

            if (rc || (*e != '\0')) {
                util_error("invalid -ADDR string '%s'", p);
                return -1;
            }

            if (offset > offset_end) {
                util_error("start ADDR $%06x > end ADDR $%06x!", offset, offset_end);
                return -1;
            }
        }
    }

    if (rc < 0) {
        util_error("invalid OFFSET string '%s'", s);
    }

    if ((offset < 0) || (offset >= 0xffffff)) {
        util_error("invalid offset %i", offset);
        rc = -1;
    }

    if (offset_out != NULL) {
        *offset_out = offset;
    }

    if (offset_end_out != NULL) {
        *offset_end_out = offset_end;
    }

    if (romh_too_out != NULL) {
        *romh_too_out = romh_too;
    }

    return rc;
}

/* ------------------------------------------------------------------------- */

static int show_all_help(int argc, char **argv)
{
    usage(1);
    return -1;
}

static int copytext(const char *s, char *d, int len, char filler)
{
    int i = 0;

    while ((i < len) && (*s != '\0')) {
        *d++ = *s++;
        ++i;
    }

    while (i < len) {
        *d++ = filler;
        ++i;
    }

    return 0;
}

static int set_outafcrt_name(int argc, char **argv)
{
    outafcrt_name = argv[0];
    util_message("Set output name to '%s'.", outafcrt_name);
    return 0;
}

static int set_description_text(int argc, char **argv)
{
    int rc;

    rc = copytext(argv[0], text_description, 16, ' ');

    if (rc >= 0) {
        util_message("Set description text to '%s'.", text_description);
    }

    return rc;
}

static int set_afusb_driver(int argc, char **argv)
{
    return afusb_select_driver(argv[0]);
}

static int set_cart_name(int argc, char **argv)
{
    int rc;

    rc = copytext(argv[0], text_cartname, 32, '\0');

    if (rc >= 0) {
        util_message("Set cart name to '%s'.", text_cartname);
    }

    return rc;
}

static int be_more_quiet(int argc, char **argv)
{
    verbosity--;
    return 0;
}

static int dummy_handleopt(int argc, char **argv)
{
    int i;

    fprintf(stderr, "opt got %i params: ", argc);

    for (i = 0; i < argc; ++i) {
        fprintf(stderr, "%s ", argv[i]);
    }

    fprintf(stderr, "\n");

    return 0;
}

typedef struct tool_options_s {
    const char name;
    const char *paramtext;
    const char *helptext;
    int num_argc;
    int (*handle)(int argc, char **argv);
} tool_options_t;

static const tool_options_t tool_options[] = {
  { 'h', NULL,
    "Show more help.",
    0,
    show_all_help
  },
  { 'o', "OUTAFCRT",
    "Save Alien Flash .crt to file OUTAFCRT. (default = use INAFCRT)",
    1,
    set_outafcrt_name
  },
  { 'd', "USBDRV  ",
    "Select Alien Flash USB driver.",
    1,
    set_afusb_driver
  },
  { 'u', "USBPAR  ",
    "Set Alien Flash USB parameter.",
    1,
    afusb_set_device_param
  },
  { 'b', "VAL     ",
    "Set USB read block size. (default = $40)",
    1,
    set_usb_read_blocksize
  },
  { 'x', "VAL     ",
    "Set USB write block size. (default = $800)",
    1,
    set_usb_write_blocksize
  },
#ifdef HAVE_AFUSB_FTDI
  { 'E', "OUTBIN  ",
    "Dump Alien Flash EEPROM to file OUTBIN.",
    1,
    afusb_set_eeprom_dump_name
  },
#endif /* HAVE_AFUSB_FTDI */
  { 'r', "VAL     ",
    "Use VAL (in hex) reads after chip erase. (default = $4000)",
    1,
    set_dummy_erase_amount
  },
  { 'w', "VAL     ",
    "Use VAL (in hex) reads after byte program. (default = $1)",
    1,
    set_dummy_prog_amount
  },
  { 'e', NULL,
    "Do erases in created AFTT files. (default = don't erase)",
    0,
    set_do_erases
  },
  { 't', "TEXT    ",
    "Use TEXT (max 16 chars) as description for .prg/.d64.",
    1,
    set_description_text
  },
  { 'n', "TEXT    ",
    "Use TEXT (max 32 chars) as cart name for created .crt.",
    1,
    set_cart_name
  },
  { 'q', NULL,
    "Be quieter.",
    0,
    be_more_quiet
  },
  { '\0', NULL, NULL, 0, dummy_handleopt }
};

static int find_option(const char *optname)
{
    int i = 0;

    if (0
        || (optname[0] != '-')
        || (optname[1] == '\0')
        || (optname[2] != '\0')) {
        return -1;
    }

    while (tool_options[i].name != '\0') {
        if ((tool_options[i].name == optname[1])) {
            break;
        }
        ++i;
    }

    if (tool_options[i].name != '\0') {
        return i;
    } else {
        return -1;
    }
}

/* ------------------------------------------------------------------------- */

int parse_options(int *argcptr, char **argv, int *i_out)
{
    int i;
    int opt_i;
    int argc = *argcptr;

    argc--;

    i = 1;

    while ((argc > 0) && ((opt_i = find_option(argv[i])) >= 0)) {
        int opt_argc = tool_options[opt_i].num_argc;

        argc--;

        if (argc < opt_argc) {
            util_error("too few arguments for OPTION '%s'!", argv[i]);
            return -1;
        }

        ++i;

        if (tool_options[opt_i].handle(argc, &(argv[i])) < 0) {
            return -1;
        }

        i += opt_argc;
        argc -= opt_argc;
    }

    *argcptr = argc;
    *i_out = i;

    return 0;
}

void usage_af_options(void)
{
    int i;

    util_message("\nAvailable OPTIONs:");

    i = 0;
    while (tool_options[i].name != '\0') {
        util_message(
                    "  -%c %s  %s",
                    tool_options[i].name,
                    tool_options[i].paramtext ? tool_options[i].paramtext : "        ",
                    tool_options[i].helptext
                    );
        ++i;
    }
}

/* ------------------------------------------------------------------------- */

void util_msleep(unsigned int msecs)
{
#ifndef IS_WINDOWS
    usleep(msecs * 1000);
#else
    Sleep(msecs);
#endif
}
